### A Pluto.jl notebook ###
# v0.20.21

using Markdown
using InteractiveUtils

# This Pluto notebook uses @bind for interactivity. When running this notebook outside of Pluto, the following 'mock version' of @bind gives bound variables a default value (instead of an error).
macro bind(def, element)
    #! format: off
    return quote
        local iv = try Base.loaded_modules[Base.PkgId(Base.UUID("6e696c72-6542-2067-7265-42206c756150"), "AbstractPlutoDingetjes")].Bonds.initial_value catch; b -> missing; end
        local el = $(esc(element))
        global $(esc(def)) = Core.applicable(Base.get, el) ? Base.get(el) : iv(el)
        el
    end
    #! format: on
end

# ╔═╡ 92c861a8-3027-4f39-a310-986aa47eea8f
begin
	using PlutoUI, Dates, SparseArrays , StatsBase, PlutoPlotly, PlotlyKaleido
	using CSV ,  DataFrames, LinearAlgebra , TimeSeries, NLsolve, LaTeXStrings
	force_pluto_mathjax_local(true)
	PlotlyKaleido.start(;mathjax = true)
end

# ╔═╡ 576d5509-527d-4c48-9f2e-007fc977a0d8
html"""
<picture>
  <source srcset="https://gci.iscte-iul.pt/2022/identidade/files/brand_v2/escolas/rgb/ibs/en/horizontal/negative/rgb_ibs_en_horizontal_negative.png" media="(prefers-color-scheme: dark)" width="240px!important" style="float:right;">
  <img src="https://gci.iscte-iul.pt/2022/identidade/files/brand_v2/escolas/rgb/ibs/en/horizontal/main/rgb_ibs_en_horizontal_main.png" width="240px" style="float:right;">
</picture>
"""

# ╔═╡ 08782891-e0f5-4c25-9657-24212101724c
md"""
# The HP filter & Impulse Response Functions

**February 23, 2026**
"""

# ╔═╡ 67198764-1378-4286-8779-8a05011bf084
md"""
##### Vivaldo Mendes
"""

# ╔═╡ 9c11028c-8ccd-4c17-a964-86f09f785ba0
md"""
---
"""

# ╔═╡ e8f80d8c-5b6d-449a-8151-7c6f057492ae
md"""
### Packages used in this notebook
"""

# ╔═╡ ad911687-8e0b-4340-81a6-452476fadf09
md"""
## 1. Functions
"""

# ╔═╡ 775afb23-1849-444c-9c3f-abd5ef688d82
md"""
!!! note 

	Students are not expected to understand all the small details of the elements that comprise the functions in this section, but they do need to know how to use them in the sections below. Students do not need to study these functions; however, they need to understand very well what they do (what they are used for).

"""

# ╔═╡ 523249f4-b761-48b8-9deb-dd72b26f6640
md"""
The functions below will be used in this notebook to compute the Hodrick-Prescott trend (HP trend). Basically, the only indispensable function is the $(html"<kbd>hp_trend</kbd>"). But to make our lives easier and more productive, we wrote the $(html"<kbd>hp_trend_df</kbd>") function, which allows us to apply the former to an entire dataframe, instead of applying it to column by column. The final function $(html"<kbd>df_arith_operations</kbd>") is used to implement simple arithemtic operations over two dataframes (like $(html"<kbd>+</kbd>") , $(html"<kbd>-</kbd>") , $(html"<kbd>\</kbd>") , etc.) , by skipping the column with the dates.

A complete understanding of the code in these three functions will not be required in this course. But students should understand how (and why) we will use them in this notebook.

"""

# ╔═╡ 25003d08-6b07-4c95-86dd-529ffbefe556
"""
This function computes the Hodrick-Prescott trend by collecting the non-trivial diagonals of matrix `A`, derived from the first order conditions of the minimization problem. It takes into  account that A is symmetric and uses the `spdiagm` function from the package `SparseArrays.jl` for such purpose.

		A = spdiagm(-2 => diag2, -1 => diag1, 0 => diag0, 1 => diag1, 2 => diag2)

- `-2 => diag2` puts the diag2 vector on the second subdiagonal (row index greater than column by 2).

- `-1 => diag1` places diag1 on the first subdiagonal.

- `0 => diag0` places diag0 on the main diagonal.

- `1 => diag1` and `2 => diag2` place the same vectors on the corresponding superdiagonals.

To compute the HP trend of, e.g., the column that represents GDP in the USdata frame, use the following code:

		GDP_trend = hp_trend(USdata.lnGDP, λ)

To compute the cycle component of GDP, just apply an arithmetic operation:

		GDP_cycles = USdata.lnGDP - GDP_trend

"""
function hp_trend(mydata::Array{Float64,1}, λ::Float64)

	n = length(mydata) 										# Number of observations
	@assert n >= 4 											# Making sure that there is enough data to apply the trend
	diag2 = λ * ones(n-2)                       			# length n-2 -> second sub/super-diagonals (±2)
	diag1 = [-2*λ; -4*λ*ones(n-3); -2*λ]  					# length n-1 -> first sub/super-diagonals (±1)
	diag0 = [1+λ; 1+5*λ; (1+6*λ)*ones(n-4); 1+5*λ; 1+λ] 	# length n -> main diagonal

	# Compact way of writting the sparse matrix A according to SparseArrays
	A = spdiagm(-2 => diag2, -1 => diag1, 0 => diag0, 1 => diag1, 2 => diag2)

	# Computing the Hodrick-Prescott trend
	A\mydata
end


# ╔═╡ f5c3e40f-38fe-45ec-b5e4-a6ea174939ee
"""

This `hp_trend_df` function uses the `hp_trend` function to compute the Hodrick-Prescott trend, but has two advantages over the latter:

- It allows the trend to be applied to an entire dataframe and producing a new data frame as the output

- It keeps the first non-numeric column (date object: year, quarter, month, etc.) of the original data frame as the first column of the new data frame.

- This is extremely convenient because we will have all the functionalities of a data frame when producing further operation on the new trended results.

To compute the HP trend of the entire dataframe (excluding the date column), use the following code:

		USdata_trends = hp_trend_df(USdata, λ)

"""
function hp_trend_df(df::DataFrame, λ::Float64)
	# Get column names
	colnames = names(df)

	# Create a new DataFrame with the date column
	trended_df = DataFrame()
	trended_df[!, colnames[1]] = df[!, colnames[1]]  # preserve date column by name

	# Apply trend to each numeric column
	for col in colnames[2:end]
	    trended_df[!, col] = hp_trend(Vector{Float64}(df[!, col]), λ)
    end

	return trended_df
end


# ╔═╡ 2663dfde-4e9a-4eb4-98c3-222c24b0f179

"""
We can pass any binary operation `+` , `-` , `*` , `\\` as a function and it will apply it column-wise, skipping the date column.


 Compute cycle component: original data - trend

		US_cycle = df_arith_operations(USdata, US_trend, -)

Compute a ratio of two data frames (skipping the date column):

	some_ratio = df_arith_operations(some_data_frame1, some_data_frame_2, /)
"""
function df_arith_operations(df1::DataFrame, df2::DataFrame, op::Function)
    colnames = names(df1)
    result_df_operations = DataFrame()

    # Preserve the date column
    result_df_operations[!, colnames[1]] = df1[!, colnames[1]]

    # Apply the operation to the remaining columns
    for col in colnames[2:end]
        result_df_operations[!, col] = op.(df1[!, col], df2[!, col])
    end

    return result_df_operations
end


# ╔═╡ 3e834ad2-bc8e-46e4-9114-2c0836955994
"""
The Hodrick-Prescott trend routine that tranposes the FOCS, step by step, into the characteristic matrix A of the minimization problem.

	A = spzeros(Float64, n, n)

represents a nxn SparseArray matrix with Floats.

"""
function hp_trend_foc(y::Vector{Float64}, λ::Float64)
    n = length(y)
    @assert n ≥ 4 "Need at least 4 observations."

    A = spzeros(Float64, n, n)

    # Left boundary conditions
    # t = 1
    A[1,1] = 1 + λ
    A[1,2] = -2λ
    A[1,3] = λ

    # t = 2
    A[2,1] = -2λ
    A[2,2] = 1 + 5λ
    A[2,3] = -4λ
    A[2,4] = λ

    # Interior conditions
    for t in 3:(n-2)
        A[t, t-2] = λ
        A[t, t-1] = -4λ
        A[t, t]   = 1 + 6λ
        A[t, t+1] = -4λ
        A[t, t+2] = λ
    end

	# Right boundary conditions
    # t = n-1
    A[n-1, n-3] = λ
    A[n-1, n-2] = -4λ
    A[n-1, n-1] = 1 + 5λ
    A[n-1, n]   = -2λ

    # t = n
    A[n, n-2] = λ
    A[n, n-1] = -2λ
    A[n, n]   = 1 + λ

    # Solve for τ
    τ = A \ y
    return τ
end;

# ╔═╡ 637413e8-c2d2-46d9-ac9c-9a57779bcc33
md"""
## 2. Data
"""

# ╔═╡ 28eff862-eb6a-4546-bfc7-f334aced0e5e
md"""
### Read the data
"""

# ╔═╡ 734242df-1633-426a-acaa-977fe4706915
md"""
To read or load the data into the notebook, we should do as usual: use the $(html"<kbd>CSV.read</kbd>") function. See the example below, where the data file that we want to load (and is located in the same folder as this notebook) is named as $(html"<kbd>my_data.csv</kbd>"), and we will name our new dataframe as "my_data":

		my_data = CSV.read("my_data.csv", DataFrame)
"""

# ╔═╡ 11943aea-98c2-4dba-aa08-ab4241a06ffc
USdata = CSV.read("Data_US.csv", DataFrame)

# ╔═╡ fd2c78a8-0ee2-481d-9f60-289dd6a009af
USdata

# ╔═╡ 094df269-1140-4d12-bf0f-b22db7fbe677
md"""
---
"""

# ╔═╡ c502d1e7-65d9-410d-96fd-79999391e8fd
md"""
### Create sub-samples of your data
"""

# ╔═╡ 51a0bef0-7b98-4513-9b83-63b1c944d9a8
md"""
If we want to work with a particular sub-period, we have to stipulate an _initial date_ and an _end date_ like this:

		my_new_data = USdata[initial date .<= USdata.Quarters .<=  end date, :]

"""

# ╔═╡ 5e99d909-c2f3-4880-9855-edc3f791370e
my_new_data = USdata[Date(1960,01,01) .<= USdata.Quarters .<= Date(1961,10,01), :] 			# Dates passed as integers
#my_new_data = USdata[Date("1960-01-01") .<= USdata.Quarters .<= Date("1961-04-01"), :] 	# Dates passed as strings

# ╔═╡ 90a061e3-58f4-45f6-ba28-22d35b2aa9d5
md"""
----
"""

# ╔═╡ e522c316-57ca-4aa8-bd58-483f96e60e94
md"""
### Doing some plotting
"""

# ╔═╡ 93a211cf-5ad7-4fdc-a2fb-6981e61a0e3a
begin
	pp1 = plot(USdata.Quarters , [USdata.CPI  USdata.UR])

	restyle!(pp1, name = ["Unemployment (𝒖)", "Inflation (𝝅)"])

	relayout!(pp1,
		height = 460,
		legend=attr(x=0.03, y=0.95), legend_bgcolor="rgba(0,0,0,0)",
		title_text= "Unemployment and inflation in the US: 1960Q1-2023Q4", title_x = 0.5, hovermode="x", font_family="Calibri", font_size=14,
		xaxis_range = ["1959-04-01" , "2024-10-01"])

	pp1
end


# ╔═╡ 964ccb85-d710-4de8-a647-5e60d21623de
md"""
##### Saving a plot as a svg, pdf, or png file.
"""

# ╔═╡ c7707bee-609b-4a8c-b154-747a279e8e21
#savefig(p_01, "az.pdf", width = 1000)

# ╔═╡ 93580fd5-721b-4183-9e10-2d512eb9e772
begin
	trace_01 = scatter(;x= USdata.Quarters, y= USdata.CPI, mode="lines",
				marker_symbol="circle", line_width= 2,
            	marker_color = "BlueViolet",  name = "Inflation (𝝅)", yaxis = "y2",
				fillcolor = "")#, text = period1_6)

	trace_02 = scatter(;x= USdata.Quarters, y= USdata.UR, mode="lines",
				marker_symbol="circle", line_width= 2,
            	marker_color = "DarkBlue", name = "Unemployment (𝒖)", yaxis = "y2",
				fillcolor = "")#, text = period1_6)

	trace_03 = scatter(;x= USdata.Quarters, y= USdata.GDP, mode="lines",
				marker_symbol="circle", marker_size="4",line_width= 2,
            	marker_color = "maroon",  name = "GDP (𝒚)") #, text = period1_6)

	layout_03 = Layout(; title_text= "GDP, Unemployment and Inflation in the US: 1960Q1-2023Q4",
	 					title_x=0.5, hovermode="x",
						#titlefont_size = 16,
						height=460,
						font_family="Calibri", font_size=14,
						legend=attr( x=0.02, y=0.94), legend_bgcolor="rgba(0,0,0,0)",

			xaxis = attr(
                		title = "Quarterly obervations",
						hovermode = "x",
						showgrid = true),
						xaxis_range = ["1959-04-01" , "2024-10-01"],

			yaxis1 = attr(
						title = " Real GDP",
						titlefont_color=  "maroon",
						tickfont_color = "maroon",
						#overlaying = "y",
						#side = "right",
						#yaxis1_tickvals = [1.5, 2.5, 3.5, 5.5],
						showgrid= false,
						zeroline=true,
						yaxis1_range=[1.5 , 5.5],
    					#dtick = 5.6 / 7
						  ),
			yaxis2 = attr(
    					title = "Unemployment , Inflation",
    					titlefont_color=  "blue",
    					tickfont_color = "blue",
    					overlaying = "y",
    					side = "right",
						showgrid = true,
						zeroline=false,
						yaxis2_range=[-25 , 5],
						#dtick = 30 / %
						  )
						)
	p_01 = plot([trace_01, trace_02, trace_03], layout_03)

end

# ╔═╡ 9e970d98-d334-4e4d-8ae8-fbd5587133e8
md"""
---
"""

# ╔═╡ d0208488-9671-4471-b90a-7eb3bb6a7623
md"""
### Inserting new columns & deleting existing columns
"""

# ╔═╡ 6f3bab31-f65f-4ca2-9e45-4cc3f630c608
md"""

To insert a new column in our dataframe $(html"<kbd>USdata</kbd>"), we can use the $(html"<kbd>insertcols!</kbd>") function from the DataFrames package. To delete or remove an existing column, we should use the $(html"<kbd>select!</kbd>") function. Next, we will provide an example of each case.

- **Inserting a new column**. Let us insert a new column into our dataframe USdata by applying the logarithm to the column that represents the data about real GDP:  $(html"<kbd>log.(USdata.GDP</kbd>"). We will label this new column as $(html"<kbd>lnGDP</kbd>"). To achieve this, we should use the following code, where the number 3 indicates the position we want for the new column (no number implies that the new column will be the last one):


			insertcols!(USdata, 3 , :lnGDP => log.(USdata.GDP))

- **Deleting/removing an existing column**. Let us remove the column labelled as $(html"<kbd>FFR</kbd>"). To do so, just apply the following code:

			select!(USdata, Not(:FFR))

Notice that our dataframe keeps the same name; the only difference is that it inherits a new column and loses an old one. Sometimes, we want to implement the operations above, but do not want their outputs displayed in the notebook to save space or keep the notebook clean. This can be done by:

- **Suppressing the output display**. Let us suppress the display of the previous code by using the semi-comma $(html"<kbd>;</kbd>") at the end of the code:

			select!(USdata, Not(:FFR));

We can always check if the operation was successful or not by typing in an empty cell the name of the dataframe we are using and running the cell:

			USdata

"""

# ╔═╡ 0f04c187-bc39-40dc-8832-d5d0f8b62120
insertcols!(USdata, 3 , :lnGDP => log.(USdata.GDP))

# ╔═╡ 809d9a65-60f8-4281-9d3a-52ba40e913b8
select!(USdata, Not(:FFR))

# ╔═╡ 3b92d30c-4de2-4c50-9b36-5c9507bc1b71
insertcols!(USdata, :lnUR => log.(USdata.UR));

# ╔═╡ df63d7a9-57da-49b6-92e1-08e10c95a475
md"""
---
"""

# ╔═╡ 61796e54-7c59-4f58-8c3b-d09d0652f074
md"""
### Creating an entirely new dataframe
"""

# ╔═╡ 37b8be8f-9327-462a-974a-2c4bf5328880
USdata2 = select(USdata, :CPI, :UR, :GDP, :lnGDP)

# ╔═╡ fdba7a56-ba94-4a1f-a235-b8ca465cf7a4
md"""
!!! note "Exercise 1"

	Insert a new column into the dataframe by computing the logarithm of M2, suppress the display of its output and check that the operation was correctly implemented.

"""

# ╔═╡ 9d9bee55-7cb2-41ce-81b8-6df112fdccec


# ╔═╡ 3da58f19-c84c-4e58-b6be-db439112e2ca
md"""
---
"""

# ╔═╡ 72cd22a6-d335-454a-9791-f9847ef649c9
md"""
## 3. The Hodrick-Prescott filter
"""

# ╔═╡ 6ee7f803-f923-4d6d-b381-8e2ea96d8772
md"""
This filter intends to separate a time-series $y_t$ into a trend $\tau_t$ and a cyclical component $\varphi_t$ such that:

$$y_t = \tau_t + \varphi_t \tag{1}$$

- The trend is the long-run component of the time series
- The cyclical component is the short-run component of the time series
- Therefore, from (1) we get:

$$\varphi_t = y_t - \tau_t \tag{2}$$

The trick is to minimize eq. (2) subject to some given constraint on $\varphi_t$. The HP trend is given by the following minimization problem with respect to $\tau_t$:

$$\min _{\tau_{t}} \left\{ {\cal{L(\tau)}} =\sum_{t=1}^{n} \underbrace{\left(y_{t}-\tau_{t}\right)^{2}}_{=\varphi_t^2} +\ {\color{red}\lambda} \sum_{t=2}^{n-1} \underbrace{\left[\left(\tau_{t+1}-\tau_{t}\right)-\left(\tau_{t}-\tau_{t-1}\right)\right]^{2}}_{\text{constraint}}\right\} \tag{3}$$

where $\lambda$ is the parameter that we set to obtain the $(HTML("<span style='color:red'>desired</span>")) smoothness in the trend.
"""

# ╔═╡ 20f855aa-2199-42b5-bbb2-8c5cfcbf6b53
md"""
---
"""

# ╔═╡ a0a4cf82-0946-490a-9b89-ad931107d619
md"""
### Setting the value of $\lambda$
"""

# ╔═╡ b5c30c98-5757-4ad9-a4c6-129b0b69e88c
md"""
`λ = `$(@bind λ Slider(0.0:800.0:150000.0, default=1600.0, show_value=true))
"""

# ╔═╡ d740937e-e748-4116-a351-633ae6630299
md"""
----
"""

# ╔═╡ d3296808-90cc-4c14-86ac-12718d1d9973
md"""
### HP trend on a single variable/column name
"""

# ╔═╡ 793a93dc-c9d7-48ec-bc64-a4a67e710158
md"""
To make calculating the HP trend very quickly understood from a computational point of view, we will first conduct the calculation for one single variable: $(html"<kbd>lnGDP</kbd>") .

Notice that logGDP is the third column in our data set "USdata". So we can simply use `USdata[:,3]` in the code below, where 3 indicates that we will perform the computation on that column, or we can use the name of that column: $(html"<kbd>USdata.lnGDP</kbd>") . This second approach is more intuitive and less prone to mistakes; for that reason, we will stick to it.

"""

# ╔═╡ d747423d-e45e-4b03-b604-f426b0ca64e4
begin
	
	GDP_trend = hp_trend( USdata.lnGDP , λ ) 	# or #GDP_trend = hp_trend(USdata[:,3], λ)	
	GDP_cycles = USdata.lnGDP - GDP_trend		# or GDP_cycles = USdata[:,3] - GDP_trend
	
end

# ╔═╡ 1092dad1-6c6a-4aef-a720-21826fa6c8f7
begin
	p10=plot(USdata.Quarters, [USdata.lnGDP  GDP_trend], Layout(height = 450, hovermode = "x"))
	relayout!(p10, Layout(#font_size = 13,
					height = 450,
					hovermode="x",
					xaxis_range = ["1959-04-01" , "2024-10-01"],
					xaxis_title = "Quarters",
	                yaxis_title = "Billion dollars (logarithmic)",
				   	title_text= "GDP versus GDP trend (US: 1960Q1-2023Q4)", title_x =0.5,
				   	titlefont_size="17"))
	restyle!(p10, name=["GDP", "HP_trend"])
	p10
end

# ╔═╡ c31fef9e-de1d-499f-a765-15f3db2a6dc0
begin
	p11=plot(USdata.Quarters, GDP_cycles, kind="bar", marker_color = "Navy")
	relayout!(p11, Layout(#font_size = 13,
						bargap = 0.4,
						height = 450,
						hovermode="x",
						xaxis_range = ["1959-04-01" , "2024-10-01"],
						xaxis_title = "Quarters",
	                	yaxis_title = "Percentage deviations",
					   	title_text= "The Output-gap: HP trended business cycles (US: 1960Q1-2023Q4)}", title_x =0.5,
					   	titlefont_size="17"))
	restyle!(p11, name=["GDP", "HP_GDP"])
	p11
end

# ╔═╡ 0f3fc6f9-f2d3-48a2-a0fb-4f8bcfddb84e
#savefig(p_09, "p09.svg", width = 900)

# ╔═╡ 9f3f14fc-aaed-4751-a054-382f978b3e6e
md"""
---
"""

# ╔═╡ 9f4c0a5a-92dc-46bd-9f6a-d980d210cb3f
md"""
### The HP trend on an entire data set
"""

# ╔═╡ 476b34bc-da4d-4dbf-9143-fb1451ea7fd7
md"""
There are two significant shortcomings in the steps we undertook above.

- Firstly, it is terribly inefficient: if we have a dataframe with, e.g., 20 variables, we must repeat the same procedure 20 times.

- Secondly, as the HP-trend can not trend non-numerical data (like dates in the first column of the original data frame: "Quarters"), this creates further problems if we want to work with dataframes (which we should).

By using the $(html"<kbd>hp_trend_df</kbd>") function instead of $(html"<kbd>hp_trend</kbd>"), the two problems above will vanish. The function $(html"<kbd>hp_trend_df</kbd>") uses the main function $(html"<kbd>hp_trend</kbd>") and has three useful properties:

- It allows the trend to be applied to an entire dataframe and produces a new dataframe as the output.

- It keeps the first non-numeric column (date object: year, quarter, month, etc.) of the original data frame as the first column of the new data frame.

- This is extremely convenient because we will have all the functionalities of a data frame when producing further operations on the new trended results.
"""

# ╔═╡ 55b995b4-84fc-445b-adcd-e3c65aea685d
md"""
To compute the HP trend associated with each variable (column) of the **entire data set**, all we need to do is to apply the following code:

		USdata_trends = hp_trend_df(USdata, λ)

"""

# ╔═╡ ae2ae896-87d0-43d6-af43-a5fada63c589
md"""
---
"""

# ╔═╡ 667a85b6-7ceb-4d99-a9b9-5f66962479d1
md"""
#### Computing the trend
"""

# ╔═╡ c05c3eb7-c0b6-4232-a2ca-21cf808a3234
USdata_trends = hp_trend_df(USdata, λ)

# ╔═╡ d349d5d5-4a68-4f67-b635-ee6ae39fc60a
md"""
---
"""

# ╔═╡ 5d9920a1-ebc8-4d2c-974a-cc9106f286f2
md"""
#### Computing the cycles
"""

# ╔═╡ 17242693-c3d4-4202-9e9e-332b5020638a
md"""

According to eq. (2), the cycles are computed as the difference between the original time series $(y_t)$ and its long term trend $(\tau_t)$:

$$\varphi_t = y_t - \tau_t \tag{2a}$$

As $y_t$ and $\tau_t$ are two dataframes (respectively,$(html"<kbd>USdata</kbd>") and $(html"<kbd>USdata_trends</kbd>")), the computation of the difference between the two dataframes (column by column) can be easily accomplished by using some simple binary operations involving the two data frames. The function $(html"<kbd>df_arith_operations</kbd>") can do that very easily, just by typing:

		USdata_cycles = df_arith_operations( USdata , USdata_trends , - )

This function will pass the binary operation $(html"<kbd>-</kbd>") as a function and it will apply it column-wise accross the two data frames, **skipping the first column (the date object)**.

"""

# ╔═╡ 3638dc40-f782-4804-8c62-d927e066e097
USdata_cycles = df_arith_operations( USdata , USdata_trends , - ) # difference between the original data and the trend data

# ╔═╡ 241cb18f-21db-4f0f-aaf6-dc88a6a0d094
md"""
---
"""

# ╔═╡ 8825b9f3-50d6-498f-b074-cd4496a9cd73
md"""
#### Some plotting
"""

# ╔═╡ 81d80d65-4f62-4936-9b8f-bef79993bd8a
let
	p11=plot(USdata.Quarters, USdata_cycles.lnGDP, kind="bar", marker_color = "Navy")
	relayout!(p11, Layout(#font_size = 13,
						bargap = 0.4,
						height = 450,
						hovermode="x",
						xaxis_range = ["1959-04-01" , "2024-10-01"],
						xaxis_title = "Quarters",
	                	yaxis_title = "Percentage deviations",
					   	title_text= "The Output-gap using the HP trend (US: 1960Q1-2023Q4)}", title_x =0.5,
					   	titlefont_size="17"))
	p11
end

# ╔═╡ f75f0124-171c-4519-abfe-e8361e95aac5
let
	p11=plot(USdata.Quarters, USdata_cycles.GDP, kind="bar", marker_color = "maroon")
	relayout!(p11, Layout(#font_size = 13,
						bargap = 0.4,
						height = 450,
						hovermode="x",
						xaxis_range = ["1959-04-01" , "2024-10-01"],
						xaxis_title = "Quarters",
	                	yaxis_title = "Percentage deviations",
					   	title_text= "Incorrect measure of the Output-gap in the US: 1960Q1-2023Q4", title_x =0.5,
					   	titlefont_size="17"))
	p11
end

# ╔═╡ 0ba2e57a-76bd-4966-9260-8b4d1d9a81fb
let
	trace_04 = scatter(; x= USdata.Quarters, y= USdata_cycles.CPI, mode = "markers+lines",
				marker_symbol = "circle", marker_size = "5.5", line_width = 0.3,
            	marker_color = "darkblue", name = "CPI")

	layout_04= Layout(;title_text= "The inflation-gap in the US: 1960Q1-2023Q4",
				height= 450,
				title_x = 0.5,
				titlefont_size = 16,
				hovermode = "x",
            	xaxis = attr(
                	title = "Quarterly obervations",
                	tickformat = "%Y",
                	hoverformat = "%Y-Q%q",
                	tick0 = "1950/04/01",
                	dtick = "M120"),

        		yaxis_title = "Percentage deviations",
				xaxis_range = ["1959-04-01" , "2024-10-01"],
        		yaxis_range=[-6, 7])

	p_02 = plot([trace_04], layout_04)
	#savefig(p_02, "p02.svg")
end

# ╔═╡ f37523c7-9171-46c8-9e81-f91343fa2a3a
let
	trace_08 = scatter(;x= USdata.Quarters, y = USdata_trends.UR, mode="lines+markers",
				marker_symbol="circle", marker_size="4.5",line_width= 0.3,
            	marker_color = "darkred", name="NUR (𝒖ⁿ)", fillcolor = "gray")

	trace_09 = scatter(;x= USdata.Quarters, y = USdata.UR, mode="lines+markers",
				marker_symbol="circle", marker_size="5",line_width= 0.3,
            	marker_color = "blue", name="Unemployment rate (𝒖)", fillcolor = "blue")

	layout_09= Layout(; height = 450,
				font_family="Calibri", font_size=15, titlefont_size = 18,
	title_text= "The Unemployment Rate vs the Natural Unemployment Rate (trend) in the US: 1960Q1-2023Q4",
				legend=attr( x=0.02, y=0.94), #legend_bgcolor="rgba(0,0,0,0)",
				title_x = 0.5, hovermode = "x",
            	xaxis = attr(
                title = "Quarterly obervations",
                tickformat = "%Y",
                hoverformat = "%Y-Q%q",
                tick0 = "1950/04/01",
                dtick = "M120"),
				xaxis_range = ["1959-04-01" , "2024-10-01"],
        		yaxis_title = " Percentage points")
        		#yaxis_range=[-10, 4])
	p_09 = plot([trace_08, trace_09], layout_09)
	#savefig(p_09, "p09.svg")
end

# ╔═╡ 966c2fdc-0737-4fb7-9bf1-28d3df6ee7e4
md"""
!!! note "Exercise 2"

	Produce a plot involving the **logarithmic** value of the Unemployment Rate and its long-term trend. Notice that you do not need to do everything from scratch. Instead, copy the code from the previous plot, and introduce the changes that are required: variable names, title, and variable labels.

"""

# ╔═╡ 9332d9d9-9591-4587-8734-0a2cf1598286


# ╔═╡ f97fa260-18d8-4f6a-94e1-46448b0d1842
md"""
!!! note "Exercise 3"

 In the slides, we saw that if a macroeconomic variable has a clear trend, the HP trend should be applied to the logarithmic value of the original observations, not to the original observations themselves.

 Using the case of real GDP, confirm that, if we divide its HP cycles (obtained without applying logarithms) by its corresponding trend (also without logarithms), we obtain a correct measure of the business cycles (numerically or graphically). By the "correct measure", we mean the cycles obtained by applying logarithms to the original observations.

"""

# ╔═╡ 45b05a24-f860-434d-8047-5b6b86a3d2c6


# ╔═╡ f0d2a99d-c287-4eb0-933d-4756a0e6a019
md"""
---
"""

# ╔═╡ 94cfd9cf-586d-4a7a-b5ee-a0cb4c3ee288
md"""
## 4. Impulse Response Functions
"""

# ╔═╡ 8c4238a8-2ca1-4a88-aee1-95c2afafb871
md"""
### AR(1)
"""

# ╔═╡ 88e1d017-2272-41a0-b97c-10ffc1c7f55b
md"""
Consider the following simple auto-regressive process of order 1 (AR(1)):

$$y_{t+1}={\color{blueviolet}a} \cdot y_{t} + \varepsilon_{t+1} \tag{1}$$

where is $a$ a parameter.

- Assume that for $t \in(1,n)$:

$$y_{1}=0 \ ; \ \varepsilon_{2}=1 \ ; \ \varepsilon_{t}=0 \ , \ \forall t\neq2$$

- This implies that at $t=2$, $y_2 = 1$, but what happens next, if there are no more shocks?
- The IRF of $y$ provides the answer.
- The dynamics of $y$ will depend crucially on the value of $\color{blueviolet}a$. Six examples:
  $$\color{blueviolet} a = \{0,0.5,0.9, 0.99, 1, 1.01\}$$

To make it easier, we start with the case of $a=0.5$. Then, we move over to all the remaining cases.
"""

# ╔═╡ 17e30d59-0856-42e1-bf47-20e6d0e37057
begin 												# this block uses normal characters
	n = 20 											# number of iterations
	y1 = 0.0 										# setting the initial condition
	l = length(y1)									# length of of the initial condition
	ε = zeros(1,n); ε[1,2] = 1 						# definig the shock
	a = 0.5 										# setting the value of a
	y = [y1  zeros(l, n-1)] 						# pre-allocation for y

		for t = 1:n-1 								# begins the "for" loop
	   		y[t+1] =  a * y[t] + ε[t+1]             # y(t+1) = a + y(t) + ε(t+1)
		end 										# ends the loop

	plot(y', mode="markers+lines",
			marker_symbol="circle", marker_size="6", line_width= 0.3, marker_color = "Blue",
			Layout(title_text = "A stochastic AR(1) process", title_x = 0.5, hovermode = "x", height = 450)
		)
end

# ╔═╡ 9ad7d089-0885-4087-9ee0-4f9c35325441
begin 														# this block uses special characters of class: \bb
	ℕ = 30 													# number of iterations
	𝕒 = (0.0, 0.5 , 0.9 , 0.99 , 1.0 , 1.01) 				# four different values for parameter α
	𝕪1 = [0.0 ; 0.0 ; 0.0 ; 0.0 ; 0.0 ; 0.0 ] 				# setting the initial conditions
	𝕃 = length(𝕪1) 											# number of initial conditions
	𝕖 = zeros(𝕃,31) 										# pre-allocation for ε
	𝕖[:,2] = [1.0 ; 1.0 ; 1.0 ; 1.0 ; 1.0 ; 1.0] 			# setting the shock
	𝕪 = [𝕪1  zeros(𝕃, ℕ-1)] 					    		# pre-allocation for 𝓎

		for 𝕜 in eachindex(𝕒) 									# nested "for" loop
			for 𝕥 = 1 : ℕ-1
				𝕪[𝕜,𝕥+1] = 𝕒[𝕜] .* 𝕪[𝕜,𝕥] .+ 𝕖[𝕜,𝕥+1]      	# y(t+1) = a + y(t) + ε(t+1), for k=6 values of a
			end
		end 												# ends the nested loop

	p101 = plot(𝕪',mode = "markers+lines", line_width="0.5", marker_size ="6")
	restyle!(p101, name=[ "𝒂 = 0" , "𝒂 = 0.5" , "𝒂 = 0.9" , "𝒂 = 0.99", "𝒂 = 1.0" , "𝒂 = 1.01" ])
	restyle!(p101, line_color = [ "blue" , "red" , "grey" , "purple" , "darkblue", ""])
	relayout!(p101, title_text = "Impulse Response Functions (IRF) from an AR(1) process", title_x = 0.5, hovermode ="x", height = 500)
	p101
end

# ╔═╡ 3311c9c9-3646-4bbb-adc5-1e09947808a1
md"""
---
"""

# ╔═╡ 32ed885e-cf2f-4cee-b8df-cc81a463d429
md"""
### AR(2)
"""

# ╔═╡ 144c6ac4-4159-4520-bddd-3b75d624a426
md"""
- Consider a more sophisticated case, an AR(2):

$$y_{t+1}= {\color{red}a} y_{t}+{\color{blueviolet}b} y_{t-1} +  \varepsilon_{t+1} \ , \quad \varepsilon_{t} \sim N(0,1) \tag{2}$$
- Assume that for $t \in(1,n)$:

$$y_{1}=0 \ ; \ y_2=0 \ ; \ \varepsilon_{3}=1 \ ; \ \varepsilon_{t}=0 \ , \ \forall t\neq 3$$

- This implies that at

  $$t=3 \ , \ \varepsilon_{3}=1 \ \Rightarrow \ y_3 = 1.$$

  What happens next, if there are no more shocks? The IRF of $y$ provides the answer.
- The dynamics of $y$ will depend on the values of $\color{blueviolet}a$ and $\color{red}b$. For simplicity consider:

$${\color{red}b = -0.9} \ ; \color{blueviolet} a = \{1.85,1.895,1.9,1.9005\}$$

"""

# ╔═╡ 7bb3d405-e175-4865-a0a0-856a69aeb145
begin 															# this block uses special characters of class: \bf
	𝐍 = 99  													# number of iterations
	𝐚 = (1.85 , 1.895 , 1.9 , 1.9005) 							# four different values for parameter a
	𝐲1 = [ 0.0  0.0 ; 0.0  0.0 ; 0.0  0.0 ; 0.0  0.0 ] 	   		# initial conditions
	𝐋 = length(𝐲1) 												# number of initial conditions
	𝛜 = zeros(𝐋-4 , 𝐍+1) 										# pre-allocation for ε
	𝛜[:,3] = [1.0 ; 1.0 ; 1.0 ; 1.0 ] 							# setting the shock
	𝐲 = [𝐲1  zeros(𝐋-4, 𝐍-1)] 					    			# pre-allocation for 𝓎

	𝐛 = -0.9 													# setting parameter b

		for k in eachindex(𝐚) 									# nested for-loop
			for t = 1 : 𝐍-1
				𝐲[k,t+2]  = 𝐚[k] .* 𝐲[k,t+1] + 𝐛 .* 𝐲[k,t] .+ 𝛜[k,t+2]
			end
		end
	p110 = plot(𝐲',mode = "markers+lines", line_width="0.5", marker_size ="5")
	restyle!(p110, name=["𝒂 = 1.85" , "𝒂 = 1.895" , "𝒂 = 1.9" , "𝒂 = 1.9005"])
	restyle!(p110, line_color =["blue" , "navy" , "gray" , "orange"])
	relayout!(p110, title_text = "Impulse Response Functions (IRF) from an AR(2) process", title_x = 0.5, hovermode = "x", height = 500, xaxis_range =[0 , 102]
	#legend=attr( x=0.995, y=0.95)
	)
	p110
end

# ╔═╡ 92866e3b-5194-4ce7-81a3-e795b1b9a9a3
md"""
---
"""

# ╔═╡ 23786f58-0f82-4a23-914c-0bb40c63f4e7
md"""
### VAR(3)
"""

# ╔═╡ 5107f7cd-d9da-472a-bcba-9f9dd04107f7
md"""
A similar reasoning can be applied to our rather more general model:

$$X_{t+1}= A + BX_{t}+ C\varepsilon_{t+1} \tag{3}$$

where $B , C$ are $n\times n$ matrices, while $X_{t+1}, X_{t}, A, \varepsilon_{t+1}$ are $n\times 1$ vectors.

Consider the following VAR(3) model:

$$X_{t+1}=\left[\begin{array}{c}
z_{t+1} \\
w_{t+1} \\
v_{t+1}
\end{array}\right] \tag{4}$$



In this example we take matrices $A, B$ and $C$ given by:

$$\begin{gathered}
A=\left[\begin{array}{c}
0.0  \\
0.0  \\
0.0
\end{array}\right], \quad

B=\left[\begin{array}{rrr}
0.97 & 0.10 & -0.05 \\
-0.3 & 0.8 & 0.05 \\
0.01 & -0.04 & 0.96
\end{array}\right] , \quad


C=\left[\begin{array}{lll}
{\color{blueviolet}1.0} & 0.0 & 0.0 \\
0.0 & 0.0 & 0.0 \\
0.0 & 0.0 & 0.0
\end{array}\right] .
\end{gathered} \tag{5}$$

- The initial state of our system (or its initial conditions) are: $z_{1}=0, w_{1}=0$ and $v_{1}=0$, that is:

$$X_{1}=[0,0,0] \tag{6}$$

- The shock only hits the variable $z_t$ $($notice the blue entry in matrix $C)$, and we assume that the shock occurs in period $t=3$.
- What happens to the dynamics of the three endogenous variables? See next figure.
"""

# ╔═╡ c1343361-86dd-447a-8ca1-9a2a506bf78d
md"""
We start by inserting the entries of matrices $A,B,C$, according to the data above, into the notebook:
"""

# ╔═╡ 62c41a14-4cdd-488c-a653-0159f10fef1d
begin
	𝐴 = zeros(3,1)											        # matrix A
	𝐵 = [ 0.97  0.10 -0.05 ; -0.3  0.8  0.05 ; 0.01 -0.04  0.96 ] 	# matrix B
	𝐶 = [ 1.0  0.0  0.0 ; 0.0  0.0  0.0 ; 0.0  0.0  0.0 ] 			# matrix C
end;

# ╔═╡ 4ae19cc8-4a3b-4118-af0a-d179f385916b
md"""
`𝑧1 = `$(@bind 𝑧1 Slider(0.0:1.0:5.0, default= 1.0, show_value=true))
"""

# ╔═╡ 7d2c6085-fd4b-42c7-a27c-b39f6ddf8290
begin 																# this block uses special characters of class: \it
	𝑁 = 40                         									# number of iterations
	𝑋1 = [0.0 ; 0.0  ; 0.0 ] 										# initial conditions
	𝐿 = length(𝑋1)													# the number of initial conditions
	𝜀 = zeros(𝐿, 𝑁) 												# pre-allocation for ε
	𝜀[1,3] = 𝑧1 													# setting the shock (using the slider above)
	𝑋 = [𝑋1  zeros(𝐿, 𝑁-1)]; 										# preallocation fro

		for t = 1 : 𝑁-1
			𝑋[:, t+1] = 𝐴 + 𝐵*𝑋[:, t] + 𝐶 * 𝜀[:, t+1]
		end


	p12 = plot(𝑋', mode = "markers+lines",
			marker_symbol = "circle", marker_size = "6", line_width = 0.3,
			Layout(title_text = "A VAR(3) process", title_x = 0.5, hovermode = "x"))
	restyle!(p12, 1:3, name = ["𝒛(𝒕)" , "𝒘(𝒕)" , "𝒗(𝒕)"])
	restyle!(p12, 1:3, line_color = ["blue" , "gray" , "navy"])
	relayout!(p12, title_text = "Impulse Response Functions (IRF) from a VAR(3) process", title_x = 0.5,
		height = 500, legend=attr( x=0.9, y=0.95), xaxis_range =[0 , 41])
	p12
	#savefig(p12, "VAR_3.svg", width = 1000)
end

# ╔═╡ 1e3710d8-8b8a-4e58-8d22-e6d5f2c5291e
md"""
---
"""

# ╔═╡ 319d1622-6458-42a4-8e87-4aae991cab61
md"""
## 5. How to produce a nice plot
"""

# ╔═╡ 9aa0ccd4-d986-4e6a-ba7b-5652e6edcc6f
begin
	# to produce a plot named pic, with Quarters in the x-axis, and CPI and UR in the y-axis, using the USdata dataframe we should start by doing:

	pic = plot(USdata.Quarters, [USdata.CPI USdata.UR])

	# Then we can add new features to our plot by using two functions in Plotly: relayout! and restyle!

	relayout!(pic, 										# changes the layout of our plot pic
		title_text = "My fancy title" , 				# adds a title title
		title_x= 0.5 ,  								# centers the title
		height = 500 , 									# allows different height values
		width = 980 , 									# allows different width values
		#hovermode = "x",								# sets the hover to the x-axis with a box
		hovermode = "x unified",  						# sets the hover to the x-axis with a bar
		xaxis_title = "Quarterly observations" ,        # adds a title to the x-axis
		yaxis_title = "Percentage points" ,           	# adds a title to the x-axis
		xaxis_range = ["1959-04-01" , "2024-10-01"],  	# allows us to change the x-axis range
		yaxis_range = [ -2 , 18 ],    					# allows us to change the y-axis range
		font_family="Calibri", 							# sets customizable font families
		font_size= 15   								# sets customizable font size
	)

	restyle!(pic, 										# adds/changes the attributes on our  plot pic
		name = ["CPI"  "UR"] , 							# name our variables as we wish
		mode = "markers+lines" , 						# allows us to set markers, lines, or markers+lines
		marker_color = ["blue"  "maroon"] ,			    # change the line colors as we wish
		marker_size =  5 , 								# allows us to use different marker sizes
		line_width = 0.5 , 								# allows us to use different line widths
	)

	pic 												# Produces the plot with the new layout and the new attributes

end

# ╔═╡ 8eb4763a-6efb-49fd-a11d-59e724dd437c
md"""
---
"""

# ╔═╡ 2fd32ff5-13f3-41cc-806e-d7e375a4c4ce
md"""
## 6. Auxiliary cells (do not delete)
"""

# ╔═╡ 243c8313-dfa2-4f96-ad95-2e9c815a15ea
TableOfContents()

# ╔═╡ fd2c9069-9b3e-4b5f-8e1a-75728c43a387
html"""
<style>
	@media screen {
		main {
			margin: 0 auto;
			max-width: 1600px;
    		padding-left: max(200px, 10%);
    		padding-right: max(383px, 10%);
            # 383px to accomodate TableOfContents(aside=true)
		}
	}
</style>
"""

# ╔═╡ efeee7f2-b450-4db9-9a09-3c88b35252af
# changes the width of the slider pane
html"""
<style>
input[type=range] {
    width: 400px;
    height: 20px;
}
</style>
"""

# ╔═╡ ca074228-b0a1-40e6-8759-10791bd93ab0
# kbd styles formatting
html"""
<style>
    kbd {
        background-color: #505050 !important;
        color: #fff !important;
        padding: 3px 6px;
        border-radius: 4px;
        font-family: monospace;
		font-size: 0.825rem !important;
        border: 1px solid #666;
    }
</style>
"""

# ╔═╡ 00000000-0000-0000-0000-000000000001
PLUTO_PROJECT_TOML_CONTENTS = """
[deps]
CSV = "336ed68f-0bac-5ca0-87d4-7b16caf5d00b"
DataFrames = "a93c6f00-e57d-5684-b7b6-d8193f3e46c0"
Dates = "ade2ca70-3891-5945-98fb-dc099432e06a"
LaTeXStrings = "b964fa9f-0449-5b57-a5c2-d3ea65f4040f"
LinearAlgebra = "37e2e46d-f89d-539d-b4ee-838fcccc9c8e"
NLsolve = "2774e3e8-f4cf-5e23-947b-6d7e65073b56"
PlotlyKaleido = "f2990250-8cf9-495f-b13a-cce12b45703c"
PlutoPlotly = "8e989ff0-3d88-8e9f-f020-2b208a939ff0"
PlutoUI = "7f904dfe-b85e-4ff6-b463-dae2292396a8"
SparseArrays = "2f01184e-e22b-5df5-ae63-d93ebab69eaf"
StatsBase = "2913bbd2-ae8a-5f71-8c99-4fb6c76f3a91"
TimeSeries = "9e3dc215-6440-5c97-bce1-76c03772f85e"

[compat]
CSV = "~0.10.15"
DataFrames = "~1.8.1"
LaTeXStrings = "~1.4.0"
NLsolve = "~4.5.1"
PlotlyKaleido = "~2.3.1"
PlutoPlotly = "~0.6.5"
PlutoUI = "~0.7.79"
StatsBase = "~0.34.10"
TimeSeries = "~0.25.2"
"""

# ╔═╡ 00000000-0000-0000-0000-000000000002
PLUTO_MANIFEST_TOML_CONTENTS = """
# This file is machine-generated - editing it directly is not advised

julia_version = "1.10.10"
manifest_format = "2.0"
project_hash = "450204e370e897a8e6206f34a42d2a8469295ea9"

[[deps.ADTypes]]
git-tree-sha1 = "f7304359109c768cf32dc5fa2d371565bb63b68a"
uuid = "47edcb42-4c32-4615-8424-f2b9edc5f35b"
version = "1.21.0"

    [deps.ADTypes.extensions]
    ADTypesChainRulesCoreExt = "ChainRulesCore"
    ADTypesConstructionBaseExt = "ConstructionBase"
    ADTypesEnzymeCoreExt = "EnzymeCore"

    [deps.ADTypes.weakdeps]
    ChainRulesCore = "d360d2e6-b24c-11e9-a2a3-2a2ae2dbcce4"
    ConstructionBase = "187b0558-2788-49d3-abe0-74a17ed4e7c9"
    EnzymeCore = "f151be2c-9106-41f4-ab19-57ee4f262869"

[[deps.AbstractPlutoDingetjes]]
deps = ["Pkg"]
git-tree-sha1 = "6e1d2a35f2f90a4bc7c2ed98079b2ba09c35b83a"
uuid = "6e696c72-6542-2067-7265-42206c756150"
version = "1.3.2"

[[deps.Adapt]]
deps = ["LinearAlgebra", "Requires"]
git-tree-sha1 = "7e35fca2bdfba44d797c53dfe63a51fabf39bfc0"
uuid = "79e6a3ab-5dfb-504d-930d-738a2a938a0e"
version = "4.4.0"

    [deps.Adapt.extensions]
    AdaptSparseArraysExt = "SparseArrays"
    AdaptStaticArraysExt = "StaticArrays"

    [deps.Adapt.weakdeps]
    SparseArrays = "2f01184e-e22b-5df5-ae63-d93ebab69eaf"
    StaticArrays = "90137ffa-7385-5640-81b9-e52037218182"

[[deps.AliasTables]]
deps = ["PtrArrays", "Random"]
git-tree-sha1 = "9876e1e164b144ca45e9e3198d0b689cadfed9ff"
uuid = "66dad0bd-aa9a-41b7-9441-69ab47430ed8"
version = "1.1.3"

[[deps.ArgTools]]
uuid = "0dad84c5-d112-42e6-8d28-ef12dabb789f"
version = "1.1.1"

[[deps.ArrayInterface]]
deps = ["Adapt", "LinearAlgebra"]
git-tree-sha1 = "d81ae5489e13bc03567d4fbbb06c546a5e53c857"
uuid = "4fba245c-0d91-5ea0-9b3e-6abc04ee57a9"
version = "7.22.0"

    [deps.ArrayInterface.extensions]
    ArrayInterfaceBandedMatricesExt = "BandedMatrices"
    ArrayInterfaceBlockBandedMatricesExt = "BlockBandedMatrices"
    ArrayInterfaceCUDAExt = "CUDA"
    ArrayInterfaceCUDSSExt = ["CUDSS", "CUDA"]
    ArrayInterfaceChainRulesCoreExt = "ChainRulesCore"
    ArrayInterfaceChainRulesExt = "ChainRules"
    ArrayInterfaceGPUArraysCoreExt = "GPUArraysCore"
    ArrayInterfaceMetalExt = "Metal"
    ArrayInterfaceReverseDiffExt = "ReverseDiff"
    ArrayInterfaceSparseArraysExt = "SparseArrays"
    ArrayInterfaceStaticArraysCoreExt = "StaticArraysCore"
    ArrayInterfaceTrackerExt = "Tracker"

    [deps.ArrayInterface.weakdeps]
    BandedMatrices = "aae01518-5342-5314-be14-df237901396f"
    BlockBandedMatrices = "ffab5731-97b5-5995-9138-79e8c1846df0"
    CUDA = "052768ef-5323-5732-b1bb-66c8b64840ba"
    CUDSS = "45b445bb-4962-46a0-9369-b4df9d0f772e"
    ChainRules = "082447d4-558c-5d27-93f4-14fc19e9eca2"
    ChainRulesCore = "d360d2e6-b24c-11e9-a2a3-2a2ae2dbcce4"
    GPUArraysCore = "46192b85-c4d5-4398-a991-12ede77f4527"
    Metal = "dde4c033-4e86-420c-a63e-0dd931031962"
    ReverseDiff = "37e2e3b7-166d-5795-8a7a-e32c996b4267"
    SparseArrays = "2f01184e-e22b-5df5-ae63-d93ebab69eaf"
    StaticArraysCore = "1e83bf80-4336-4d27-bf5d-d5a4f845583c"
    Tracker = "9f7883ad-71c0-57eb-9f7f-b5c9e6d3789c"

[[deps.Artifacts]]
uuid = "56f22d72-fd6d-98f1-02f0-08ddc0907c33"

[[deps.Base64]]
uuid = "2a0f44e3-6c83-55bd-87e4-b1978d98bd5f"

[[deps.CSV]]
deps = ["CodecZlib", "Dates", "FilePathsBase", "InlineStrings", "Mmap", "Parsers", "PooledArrays", "PrecompileTools", "SentinelArrays", "Tables", "Unicode", "WeakRefStrings", "WorkerUtilities"]
git-tree-sha1 = "deddd8725e5e1cc49ee205a1964256043720a6c3"
uuid = "336ed68f-0bac-5ca0-87d4-7b16caf5d00b"
version = "0.10.15"

[[deps.CodecZlib]]
deps = ["TranscodingStreams", "Zlib_jll"]
git-tree-sha1 = "962834c22b66e32aa10f7611c08c8ca4e20749a9"
uuid = "944b1d66-785c-5afd-91f1-9de20f533193"
version = "0.7.8"

[[deps.ColorSchemes]]
deps = ["ColorTypes", "ColorVectorSpace", "Colors", "FixedPointNumbers", "PrecompileTools", "Random"]
git-tree-sha1 = "b0fd3f56fa442f81e0a47815c92245acfaaa4e34"
uuid = "35d6a980-a343-548e-a6ea-1d62b119f2f4"
version = "3.31.0"

[[deps.ColorTypes]]
deps = ["FixedPointNumbers", "Random"]
git-tree-sha1 = "67e11ee83a43eb71ddc950302c53bf33f0690dfe"
uuid = "3da002f7-5984-5a60-b8a6-cbb66c0b333f"
version = "0.12.1"

    [deps.ColorTypes.extensions]
    StyledStringsExt = "StyledStrings"

    [deps.ColorTypes.weakdeps]
    StyledStrings = "f489334b-da3d-4c2e-b8f0-e476e12c162b"

[[deps.ColorVectorSpace]]
deps = ["ColorTypes", "FixedPointNumbers", "LinearAlgebra", "Requires", "Statistics", "TensorCore"]
git-tree-sha1 = "8b3b6f87ce8f65a2b4f857528fd8d70086cd72b1"
uuid = "c3611d14-8923-5661-9e6a-0046d554d3a4"
version = "0.11.0"
weakdeps = ["SpecialFunctions"]

    [deps.ColorVectorSpace.extensions]
    SpecialFunctionsExt = "SpecialFunctions"

[[deps.Colors]]
deps = ["ColorTypes", "FixedPointNumbers", "Reexport"]
git-tree-sha1 = "37ea44092930b1811e666c3bc38065d7d87fcc74"
uuid = "5ae59095-9a9b-59fe-a467-6f913c188581"
version = "0.13.1"

[[deps.CommonSubexpressions]]
deps = ["MacroTools"]
git-tree-sha1 = "cda2cfaebb4be89c9084adaca7dd7333369715c5"
uuid = "bbf7d656-a473-5ed7-a52c-81e309532950"
version = "0.3.1"

[[deps.Compat]]
deps = ["TOML", "UUIDs"]
git-tree-sha1 = "9d8a54ce4b17aa5bdce0ea5c34bc5e7c340d16ad"
uuid = "34da2185-b29b-5c13-b0c7-acf172513d20"
version = "4.18.1"
weakdeps = ["Dates", "LinearAlgebra"]

    [deps.Compat.extensions]
    CompatLinearAlgebraExt = "LinearAlgebra"

[[deps.CompilerSupportLibraries_jll]]
deps = ["Artifacts", "Libdl"]
uuid = "e66e0078-7015-5450-92f7-15fbd957f2ae"
version = "1.1.1+0"

[[deps.ConstructionBase]]
git-tree-sha1 = "b4b092499347b18a015186eae3042f72267106cb"
uuid = "187b0558-2788-49d3-abe0-74a17ed4e7c9"
version = "1.6.0"

    [deps.ConstructionBase.extensions]
    ConstructionBaseIntervalSetsExt = "IntervalSets"
    ConstructionBaseLinearAlgebraExt = "LinearAlgebra"
    ConstructionBaseStaticArraysExt = "StaticArrays"

    [deps.ConstructionBase.weakdeps]
    IntervalSets = "8197267c-284f-5f27-9208-e0e47529a953"
    LinearAlgebra = "37e2e46d-f89d-539d-b4ee-838fcccc9c8e"
    StaticArrays = "90137ffa-7385-5640-81b9-e52037218182"

[[deps.Crayons]]
git-tree-sha1 = "249fe38abf76d48563e2f4556bebd215aa317e15"
uuid = "a8cc5b0e-0ffa-5ad4-8c14-923d3ee1735f"
version = "4.1.1"

[[deps.DataAPI]]
git-tree-sha1 = "abe83f3a2f1b857aac70ef8b269080af17764bbe"
uuid = "9a962f9c-6df0-11e9-0e5d-c546b8b5ee8a"
version = "1.16.0"

[[deps.DataFrames]]
deps = ["Compat", "DataAPI", "DataStructures", "Future", "InlineStrings", "InvertedIndices", "IteratorInterfaceExtensions", "LinearAlgebra", "Markdown", "Missings", "PooledArrays", "PrecompileTools", "PrettyTables", "Printf", "Random", "Reexport", "SentinelArrays", "SortingAlgorithms", "Statistics", "TableTraits", "Tables", "Unicode"]
git-tree-sha1 = "d8928e9169ff76c6281f39a659f9bca3a573f24c"
uuid = "a93c6f00-e57d-5684-b7b6-d8193f3e46c0"
version = "1.8.1"

[[deps.DataStructures]]
deps = ["OrderedCollections"]
git-tree-sha1 = "e357641bb3e0638d353c4b29ea0e40ea644066a6"
uuid = "864edb3b-99cc-5e75-8d2d-829cb0a9cfe8"
version = "0.19.3"

[[deps.DataValueInterfaces]]
git-tree-sha1 = "bfc1187b79289637fa0ef6d4436ebdfe6905cbd6"
uuid = "e2d170a0-9d28-54be-80f0-106bbe20a464"
version = "1.0.0"

[[deps.Dates]]
deps = ["Printf"]
uuid = "ade2ca70-3891-5945-98fb-dc099432e06a"

[[deps.DelimitedFiles]]
deps = ["Mmap"]
git-tree-sha1 = "9e2f36d3c96a820c678f2f1f1782582fcf685bae"
uuid = "8bb1440f-4735-579b-a4ab-409b98df4dab"
version = "1.9.1"

[[deps.DiffResults]]
deps = ["StaticArraysCore"]
git-tree-sha1 = "782dd5f4561f5d267313f23853baaaa4c52ea621"
uuid = "163ba53b-c6d8-5494-b064-1a9d43ac40c5"
version = "1.1.0"

[[deps.DiffRules]]
deps = ["IrrationalConstants", "LogExpFunctions", "NaNMath", "Random", "SpecialFunctions"]
git-tree-sha1 = "23163d55f885173722d1e4cf0f6110cdbaf7e272"
uuid = "b552c78f-8df3-52c6-915a-8e097449b14b"
version = "1.15.1"

[[deps.DifferentiationInterface]]
deps = ["ADTypes", "LinearAlgebra"]
git-tree-sha1 = "7ae99144ea44715402c6c882bfef2adbeadbc4ce"
uuid = "a0c0ee7d-e4b9-4e03-894e-1c5f64a51d63"
version = "0.7.16"

    [deps.DifferentiationInterface.extensions]
    DifferentiationInterfaceChainRulesCoreExt = "ChainRulesCore"
    DifferentiationInterfaceDiffractorExt = "Diffractor"
    DifferentiationInterfaceEnzymeExt = ["EnzymeCore", "Enzyme"]
    DifferentiationInterfaceFastDifferentiationExt = "FastDifferentiation"
    DifferentiationInterfaceFiniteDiffExt = "FiniteDiff"
    DifferentiationInterfaceFiniteDifferencesExt = "FiniteDifferences"
    DifferentiationInterfaceForwardDiffExt = ["ForwardDiff", "DiffResults"]
    DifferentiationInterfaceGPUArraysCoreExt = "GPUArraysCore"
    DifferentiationInterfaceGTPSAExt = "GTPSA"
    DifferentiationInterfaceMooncakeExt = "Mooncake"
    DifferentiationInterfacePolyesterForwardDiffExt = ["PolyesterForwardDiff", "ForwardDiff", "DiffResults"]
    DifferentiationInterfaceReverseDiffExt = ["ReverseDiff", "DiffResults"]
    DifferentiationInterfaceSparseArraysExt = "SparseArrays"
    DifferentiationInterfaceSparseConnectivityTracerExt = "SparseConnectivityTracer"
    DifferentiationInterfaceSparseMatrixColoringsExt = "SparseMatrixColorings"
    DifferentiationInterfaceStaticArraysExt = "StaticArrays"
    DifferentiationInterfaceSymbolicsExt = "Symbolics"
    DifferentiationInterfaceTrackerExt = "Tracker"
    DifferentiationInterfaceZygoteExt = ["Zygote", "ForwardDiff"]

    [deps.DifferentiationInterface.weakdeps]
    ChainRulesCore = "d360d2e6-b24c-11e9-a2a3-2a2ae2dbcce4"
    DiffResults = "163ba53b-c6d8-5494-b064-1a9d43ac40c5"
    Diffractor = "9f5e2b26-1114-432f-b630-d3fe2085c51c"
    Enzyme = "7da242da-08ed-463a-9acd-ee780be4f1d9"
    EnzymeCore = "f151be2c-9106-41f4-ab19-57ee4f262869"
    FastDifferentiation = "eb9bf01b-bf85-4b60-bf87-ee5de06c00be"
    FiniteDiff = "6a86dc24-6348-571c-b903-95158fe2bd41"
    FiniteDifferences = "26cc04aa-876d-5657-8c51-4c34ba976000"
    ForwardDiff = "f6369f11-7733-5829-9624-2563aa707210"
    GPUArraysCore = "46192b85-c4d5-4398-a991-12ede77f4527"
    GTPSA = "b27dd330-f138-47c5-815b-40db9dd9b6e8"
    Mooncake = "da2b9cff-9c12-43a0-ae48-6db2b0edb7d6"
    PolyesterForwardDiff = "98d1487c-24ca-40b6-b7ab-df2af84e126b"
    ReverseDiff = "37e2e3b7-166d-5795-8a7a-e32c996b4267"
    SparseArrays = "2f01184e-e22b-5df5-ae63-d93ebab69eaf"
    SparseConnectivityTracer = "9f842d2f-2579-4b1d-911e-f412cf18a3f5"
    SparseMatrixColorings = "0a514795-09f3-496d-8182-132a7b665d35"
    StaticArrays = "90137ffa-7385-5640-81b9-e52037218182"
    Symbolics = "0c5d862f-8b57-4792-8d23-62f2024744c7"
    Tracker = "9f7883ad-71c0-57eb-9f7f-b5c9e6d3789c"
    Zygote = "e88e6eb3-aa80-5325-afca-941959d7151f"

[[deps.Distances]]
deps = ["LinearAlgebra", "Statistics", "StatsAPI"]
git-tree-sha1 = "c7e3a542b999843086e2f29dac96a618c105be1d"
uuid = "b4f34e82-e78d-54a5-968a-f98e89d6e8f7"
version = "0.10.12"

    [deps.Distances.extensions]
    DistancesChainRulesCoreExt = "ChainRulesCore"
    DistancesSparseArraysExt = "SparseArrays"

    [deps.Distances.weakdeps]
    ChainRulesCore = "d360d2e6-b24c-11e9-a2a3-2a2ae2dbcce4"
    SparseArrays = "2f01184e-e22b-5df5-ae63-d93ebab69eaf"

[[deps.Distributed]]
deps = ["Random", "Serialization", "Sockets"]
uuid = "8ba89e20-285c-5b6f-9357-94700520ee1b"

[[deps.DocStringExtensions]]
git-tree-sha1 = "7442a5dfe1ebb773c29cc2962a8980f47221d76c"
uuid = "ffbed154-4ef7-542d-bbb7-c09d3a79fcae"
version = "0.9.5"

[[deps.Downloads]]
deps = ["ArgTools", "FileWatching", "LibCURL", "NetworkOptions"]
uuid = "f43a241f-c20a-4ad4-852c-f6b1247861c6"
version = "1.6.0"

[[deps.FilePathsBase]]
deps = ["Compat", "Dates"]
git-tree-sha1 = "3bab2c5aa25e7840a4b065805c0cdfc01f3068d2"
uuid = "48062228-2e41-5def-b9a4-89aafe57970f"
version = "0.9.24"
weakdeps = ["Mmap", "Test"]

    [deps.FilePathsBase.extensions]
    FilePathsBaseMmapExt = "Mmap"
    FilePathsBaseTestExt = "Test"

[[deps.FileWatching]]
uuid = "7b1f6079-737a-58dc-b8bc-7a2ca5c1b5ee"

[[deps.FiniteDiff]]
deps = ["ArrayInterface", "LinearAlgebra", "Setfield"]
git-tree-sha1 = "9340ca07ca27093ff68418b7558ca37b05f8aeb1"
uuid = "6a86dc24-6348-571c-b903-95158fe2bd41"
version = "2.29.0"

    [deps.FiniteDiff.extensions]
    FiniteDiffBandedMatricesExt = "BandedMatrices"
    FiniteDiffBlockBandedMatricesExt = "BlockBandedMatrices"
    FiniteDiffSparseArraysExt = "SparseArrays"
    FiniteDiffStaticArraysExt = "StaticArrays"

    [deps.FiniteDiff.weakdeps]
    BandedMatrices = "aae01518-5342-5314-be14-df237901396f"
    BlockBandedMatrices = "ffab5731-97b5-5995-9138-79e8c1846df0"
    SparseArrays = "2f01184e-e22b-5df5-ae63-d93ebab69eaf"
    StaticArrays = "90137ffa-7385-5640-81b9-e52037218182"

[[deps.FixedPointNumbers]]
deps = ["Statistics"]
git-tree-sha1 = "05882d6995ae5c12bb5f36dd2ed3f61c98cbb172"
uuid = "53c48c17-4a7d-5ca2-90c5-79b7896eea93"
version = "0.8.5"

[[deps.ForwardDiff]]
deps = ["CommonSubexpressions", "DiffResults", "DiffRules", "LinearAlgebra", "LogExpFunctions", "NaNMath", "Preferences", "Printf", "Random", "SpecialFunctions"]
git-tree-sha1 = "eef4c86803f47dcb61e9b8790ecaa96956fdd8ae"
uuid = "f6369f11-7733-5829-9624-2563aa707210"
version = "1.3.2"

    [deps.ForwardDiff.extensions]
    ForwardDiffStaticArraysExt = "StaticArrays"

    [deps.ForwardDiff.weakdeps]
    StaticArrays = "90137ffa-7385-5640-81b9-e52037218182"

[[deps.Future]]
deps = ["Random"]
uuid = "9fa8497b-333b-5362-9e8d-4d0656e87820"

[[deps.HashArrayMappedTries]]
git-tree-sha1 = "2eaa69a7cab70a52b9687c8bf950a5a93ec895ae"
uuid = "076d061b-32b6-4027-95e0-9a2c6f6d7e74"
version = "0.2.0"

[[deps.Hyperscript]]
deps = ["Test"]
git-tree-sha1 = "179267cfa5e712760cd43dcae385d7ea90cc25a4"
uuid = "47d2ed2b-36de-50cf-bf87-49c2cf4b8b91"
version = "0.0.5"

[[deps.HypertextLiteral]]
deps = ["Tricks"]
git-tree-sha1 = "7134810b1afce04bbc1045ca1985fbe81ce17653"
uuid = "ac1192a8-f4b3-4bfe-ba22-af5b92cd3ab2"
version = "0.9.5"

[[deps.IOCapture]]
deps = ["Logging", "Random"]
git-tree-sha1 = "0ee181ec08df7d7c911901ea38baf16f755114dc"
uuid = "b5f81e59-6552-4d32-b1f0-c071b021bf89"
version = "1.0.0"

[[deps.InlineStrings]]
git-tree-sha1 = "8f3d257792a522b4601c24a577954b0a8cd7334d"
uuid = "842dd82b-1e85-43dc-bf29-5d0ee9dffc48"
version = "1.4.5"

    [deps.InlineStrings.extensions]
    ArrowTypesExt = "ArrowTypes"
    ParsersExt = "Parsers"

    [deps.InlineStrings.weakdeps]
    ArrowTypes = "31f734f8-188a-4ce0-8406-c8a06bd891cd"
    Parsers = "69de0a69-1ddd-5017-9359-2bf0b02dc9f0"

[[deps.InteractiveUtils]]
deps = ["Markdown"]
uuid = "b77e0a4c-d291-57a0-90e8-8db25a27a240"

[[deps.InvertedIndices]]
git-tree-sha1 = "6da3c4316095de0f5ee2ebd875df8721e7e0bdbe"
uuid = "41ab1584-1d38-5bbf-9106-f11c6c58b48f"
version = "1.3.1"

[[deps.IrrationalConstants]]
git-tree-sha1 = "b2d91fe939cae05960e760110b328288867b5758"
uuid = "92d709cd-6900-40b7-9082-c6be49f344b6"
version = "0.2.6"

[[deps.IteratorInterfaceExtensions]]
git-tree-sha1 = "a3f24677c21f5bbe9d2a714f95dcd58337fb2856"
uuid = "82899510-4779-5014-852e-03e436cf321d"
version = "1.0.0"

[[deps.JLLWrappers]]
deps = ["Artifacts", "Preferences"]
git-tree-sha1 = "0533e564aae234aff59ab625543145446d8b6ec2"
uuid = "692b3bcd-3c85-4b1f-b108-f13ce0eb3210"
version = "1.7.1"

[[deps.JSON]]
deps = ["Dates", "Logging", "Parsers", "PrecompileTools", "StructUtils", "UUIDs", "Unicode"]
git-tree-sha1 = "b3ad4a0255688dcb895a52fafbaae3023b588a90"
uuid = "682c06a0-de6a-54ab-a142-c8b1cf79cde6"
version = "1.4.0"

    [deps.JSON.extensions]
    JSONArrowExt = ["ArrowTypes"]

    [deps.JSON.weakdeps]
    ArrowTypes = "31f734f8-188a-4ce0-8406-c8a06bd891cd"

[[deps.Kaleido_jll]]
deps = ["Artifacts", "JLLWrappers", "Libdl", "Pkg"]
git-tree-sha1 = "43032da5832754f58d14a91ffbe86d5f176acda9"
uuid = "f7e6163d-2fa5-5f23-b69c-1db539e41963"
version = "0.2.1+0"

[[deps.LaTeXStrings]]
git-tree-sha1 = "dda21b8cbd6a6c40d9d02a73230f9d70fed6918c"
uuid = "b964fa9f-0449-5b57-a5c2-d3ea65f4040f"
version = "1.4.0"

[[deps.LibCURL]]
deps = ["LibCURL_jll", "MozillaCACerts_jll"]
uuid = "b27032c2-a3e7-50c8-80cd-2d36dbcbfd21"
version = "0.6.4"

[[deps.LibCURL_jll]]
deps = ["Artifacts", "LibSSH2_jll", "Libdl", "MbedTLS_jll", "Zlib_jll", "nghttp2_jll"]
uuid = "deac9b47-8bc7-5906-a0fe-35ac56dc84c0"
version = "8.4.0+0"

[[deps.LibGit2]]
deps = ["Base64", "LibGit2_jll", "NetworkOptions", "Printf", "SHA"]
uuid = "76f85450-5226-5b5a-8eaa-529ad045b433"

[[deps.LibGit2_jll]]
deps = ["Artifacts", "LibSSH2_jll", "Libdl", "MbedTLS_jll"]
uuid = "e37daf67-58a4-590a-8e99-b0245dd2ffc5"
version = "1.6.4+0"

[[deps.LibSSH2_jll]]
deps = ["Artifacts", "Libdl", "MbedTLS_jll"]
uuid = "29816b5a-b9ab-546f-933c-edad1886dfa8"
version = "1.11.0+1"

[[deps.Libdl]]
uuid = "8f399da3-3557-5675-b5ff-fb832c97cbdb"

[[deps.LineSearches]]
deps = ["LinearAlgebra", "NLSolversBase", "NaNMath", "Printf"]
git-tree-sha1 = "9ea3422d03222c6de679934d1c08f0a99405aa03"
uuid = "d3d80556-e9d4-5f37-9878-2ab0fcc64255"
version = "7.5.1"

[[deps.LinearAlgebra]]
deps = ["Libdl", "OpenBLAS_jll", "libblastrampoline_jll"]
uuid = "37e2e46d-f89d-539d-b4ee-838fcccc9c8e"

[[deps.LogExpFunctions]]
deps = ["DocStringExtensions", "IrrationalConstants", "LinearAlgebra"]
git-tree-sha1 = "13ca9e2586b89836fd20cccf56e57e2b9ae7f38f"
uuid = "2ab3a3ac-af41-5b50-aa03-7779005ae688"
version = "0.3.29"

    [deps.LogExpFunctions.extensions]
    LogExpFunctionsChainRulesCoreExt = "ChainRulesCore"
    LogExpFunctionsChangesOfVariablesExt = "ChangesOfVariables"
    LogExpFunctionsInverseFunctionsExt = "InverseFunctions"

    [deps.LogExpFunctions.weakdeps]
    ChainRulesCore = "d360d2e6-b24c-11e9-a2a3-2a2ae2dbcce4"
    ChangesOfVariables = "9e997f8a-9a97-42d5-a9f1-ce6bfc15e2c0"
    InverseFunctions = "3587e190-3f89-42d0-90ee-14403ec27112"

[[deps.Logging]]
uuid = "56ddb016-857b-54e1-b83d-db4d58db5568"

[[deps.MIMEs]]
git-tree-sha1 = "c64d943587f7187e751162b3b84445bbbd79f691"
uuid = "6c6e2e6c-3030-632d-7369-2d6c69616d65"
version = "1.1.0"

[[deps.MacroTools]]
git-tree-sha1 = "1e0228a030642014fe5cfe68c2c0a818f9e3f522"
uuid = "1914dd2f-81c6-5fcd-8719-6d5c9610ff09"
version = "0.5.16"

[[deps.Markdown]]
deps = ["Base64"]
uuid = "d6f4376e-aef5-505a-96c1-9c027394607a"

[[deps.MbedTLS_jll]]
deps = ["Artifacts", "Libdl"]
uuid = "c8ffd9c3-330d-5841-b78e-0817d7145fa1"
version = "2.28.2+1"

[[deps.Missings]]
deps = ["DataAPI"]
git-tree-sha1 = "ec4f7fbeab05d7747bdf98eb74d130a2a2ed298d"
uuid = "e1d29d7a-bbdc-5cf2-9ac0-f12de2c33e28"
version = "1.2.0"

[[deps.Mmap]]
uuid = "a63ad114-7e13-5084-954f-fe012c677804"

[[deps.MozillaCACerts_jll]]
uuid = "14a3606d-f60d-562e-9121-12d972cd8159"
version = "2023.1.10"

[[deps.NLSolversBase]]
deps = ["ADTypes", "DifferentiationInterface", "Distributed", "FiniteDiff", "ForwardDiff"]
git-tree-sha1 = "25a6638571a902ecfb1ae2a18fc1575f86b1d4df"
uuid = "d41bc354-129a-5804-8e4c-c37616107c6c"
version = "7.10.0"

[[deps.NLsolve]]
deps = ["Distances", "LineSearches", "LinearAlgebra", "NLSolversBase", "Printf", "Reexport"]
git-tree-sha1 = "019f12e9a1a7880459d0173c182e6a99365d7ac1"
uuid = "2774e3e8-f4cf-5e23-947b-6d7e65073b56"
version = "4.5.1"

[[deps.NaNMath]]
deps = ["OpenLibm_jll"]
git-tree-sha1 = "9b8215b1ee9e78a293f99797cd31375471b2bcae"
uuid = "77ba4419-2d1f-58cd-9bb1-8ffee604a2e3"
version = "1.1.3"

[[deps.NetworkOptions]]
uuid = "ca575930-c2e3-43a9-ace4-1e988b2c1908"
version = "1.2.0"

[[deps.OpenBLAS_jll]]
deps = ["Artifacts", "CompilerSupportLibraries_jll", "Libdl"]
uuid = "4536629a-c528-5b80-bd46-f80d51c5b363"
version = "0.3.23+4"

[[deps.OpenLibm_jll]]
deps = ["Artifacts", "Libdl"]
uuid = "05823500-19ac-5b8b-9628-191a04bc5112"
version = "0.8.5+0"

[[deps.OpenSpecFun_jll]]
deps = ["Artifacts", "CompilerSupportLibraries_jll", "JLLWrappers", "Libdl"]
git-tree-sha1 = "1346c9208249809840c91b26703912dff463d335"
uuid = "efe28fd5-8261-553b-a9e1-b2916fc3738e"
version = "0.5.6+0"

[[deps.OrderedCollections]]
git-tree-sha1 = "05868e21324cede2207c6f0f466b4bfef6d5e7ee"
uuid = "bac558e1-5e72-5ebc-8fee-abe8a469f55d"
version = "1.8.1"

[[deps.Parameters]]
deps = ["OrderedCollections", "UnPack"]
git-tree-sha1 = "34c0e9ad262e5f7fc75b10a9952ca7692cfc5fbe"
uuid = "d96e819e-fc66-5662-9728-84c9c7592b0a"
version = "0.12.3"

[[deps.Parsers]]
deps = ["Dates", "PrecompileTools", "UUIDs"]
git-tree-sha1 = "7d2f8f21da5db6a806faf7b9b292296da42b2810"
uuid = "69de0a69-1ddd-5017-9359-2bf0b02dc9f0"
version = "2.8.3"

[[deps.Pkg]]
deps = ["Artifacts", "Dates", "Downloads", "FileWatching", "LibGit2", "Libdl", "Logging", "Markdown", "Printf", "REPL", "Random", "SHA", "Serialization", "TOML", "Tar", "UUIDs", "p7zip_jll"]
uuid = "44cfe95a-1eb2-52ea-b672-e2afdf69b78f"
version = "1.10.0"

[[deps.PlotlyBase]]
deps = ["ColorSchemes", "Colors", "Dates", "DelimitedFiles", "DocStringExtensions", "JSON", "LaTeXStrings", "Logging", "Parameters", "Pkg", "REPL", "Requires", "Statistics", "UUIDs"]
git-tree-sha1 = "6256ab3ee24ef079b3afa310593817e069925eeb"
uuid = "a03496cd-edff-5a9b-9e67-9cda94a718b5"
version = "0.8.23"

    [deps.PlotlyBase.extensions]
    DataFramesExt = "DataFrames"
    DistributionsExt = "Distributions"
    IJuliaExt = "IJulia"
    JSON3Ext = "JSON3"

    [deps.PlotlyBase.weakdeps]
    DataFrames = "a93c6f00-e57d-5684-b7b6-d8193f3e46c0"
    Distributions = "31c24e10-a181-5473-b8eb-7969acd0382f"
    IJulia = "7073ff75-c697-5162-941a-fcdaad2a7d2a"
    JSON3 = "0f8b85d8-7281-11e9-16c2-39a750bddbf1"

[[deps.PlotlyKaleido]]
deps = ["Artifacts", "Base64", "JSON", "Kaleido_jll"]
git-tree-sha1 = "c7a270d11881c0709052f8e3def3d60d551e99f3"
uuid = "f2990250-8cf9-495f-b13a-cce12b45703c"
version = "2.3.1"

[[deps.PlutoPlotly]]
deps = ["AbstractPlutoDingetjes", "Artifacts", "ColorSchemes", "Colors", "Dates", "Downloads", "HypertextLiteral", "InteractiveUtils", "LaTeXStrings", "Markdown", "Pkg", "PlotlyBase", "PrecompileTools", "Reexport", "ScopedValues", "Scratch", "TOML"]
git-tree-sha1 = "8acd04abc9a636ef57004f4c2e6f3f6ed4611099"
uuid = "8e989ff0-3d88-8e9f-f020-2b208a939ff0"
version = "0.6.5"

    [deps.PlutoPlotly.extensions]
    PlotlyKaleidoExt = "PlotlyKaleido"
    UnitfulExt = "Unitful"

    [deps.PlutoPlotly.weakdeps]
    PlotlyKaleido = "f2990250-8cf9-495f-b13a-cce12b45703c"
    Unitful = "1986cc42-f94f-5a68-af5c-568840ba703d"

[[deps.PlutoUI]]
deps = ["AbstractPlutoDingetjes", "Base64", "ColorTypes", "Dates", "Downloads", "FixedPointNumbers", "Hyperscript", "HypertextLiteral", "IOCapture", "InteractiveUtils", "Logging", "MIMEs", "Markdown", "Random", "Reexport", "URIs", "UUIDs"]
git-tree-sha1 = "3ac7038a98ef6977d44adeadc73cc6f596c08109"
uuid = "7f904dfe-b85e-4ff6-b463-dae2292396a8"
version = "0.7.79"

[[deps.PooledArrays]]
deps = ["DataAPI", "Future"]
git-tree-sha1 = "36d8b4b899628fb92c2749eb488d884a926614d3"
uuid = "2dfb63ee-cc39-5dd5-95bd-886bf059d720"
version = "1.4.3"

[[deps.PrecompileTools]]
deps = ["Preferences"]
git-tree-sha1 = "5aa36f7049a63a1528fe8f7c3f2113413ffd4e1f"
uuid = "aea7be01-6a6a-4083-8856-8a6e6704d82a"
version = "1.2.1"

[[deps.Preferences]]
deps = ["TOML"]
git-tree-sha1 = "522f093a29b31a93e34eaea17ba055d850edea28"
uuid = "21216c6a-2e73-6563-6e65-726566657250"
version = "1.5.1"

[[deps.PrettyTables]]
deps = ["Crayons", "LaTeXStrings", "Markdown", "PrecompileTools", "Printf", "REPL", "Reexport", "StringManipulation", "Tables"]
git-tree-sha1 = "211530a7dc76ab59087f4d4d1fc3f086fbe87594"
uuid = "08abe8d2-0d0c-5749-adfa-8a2ac140af0d"
version = "3.2.3"

    [deps.PrettyTables.extensions]
    PrettyTablesTypstryExt = "Typstry"

    [deps.PrettyTables.weakdeps]
    Typstry = "f0ed7684-a786-439e-b1e3-3b82803b501e"

[[deps.Printf]]
deps = ["Unicode"]
uuid = "de0858da-6303-5e67-8744-51eddeeeb8d7"

[[deps.PtrArrays]]
git-tree-sha1 = "1d36ef11a9aaf1e8b74dacc6a731dd1de8fd493d"
uuid = "43287f4e-b6f4-7ad1-bb20-aadabca52c3d"
version = "1.3.0"

[[deps.REPL]]
deps = ["InteractiveUtils", "Markdown", "Sockets", "Unicode"]
uuid = "3fa0cd96-eef1-5676-8a61-b3b8758bbffb"

[[deps.Random]]
deps = ["SHA"]
uuid = "9a3f8284-a2c9-5f02-9a11-845980a1fd5c"

[[deps.RecipesBase]]
deps = ["PrecompileTools"]
git-tree-sha1 = "5c3d09cc4f31f5fc6af001c250bf1278733100ff"
uuid = "3cdcf5f2-1ef4-517c-9805-6587b60abb01"
version = "1.3.4"

[[deps.Reexport]]
git-tree-sha1 = "45e428421666073eab6f2da5c9d310d99bb12f9b"
uuid = "189a3867-3050-52da-a836-e630ba90ab69"
version = "1.2.2"

[[deps.Requires]]
deps = ["UUIDs"]
git-tree-sha1 = "62389eeff14780bfe55195b7204c0d8738436d64"
uuid = "ae029012-a4dd-5104-9daa-d747884805df"
version = "1.3.1"

[[deps.SHA]]
uuid = "ea8e919c-243c-51af-8825-aaa63cd721ce"
version = "0.7.0"

[[deps.ScopedValues]]
deps = ["HashArrayMappedTries", "Logging"]
git-tree-sha1 = "c3b2323466378a2ba15bea4b2f73b081e022f473"
uuid = "7e506255-f358-4e82-b7e4-beb19740aa63"
version = "1.5.0"

[[deps.Scratch]]
deps = ["Dates"]
git-tree-sha1 = "9b81b8393e50b7d4e6d0a9f14e192294d3b7c109"
uuid = "6c6a2e73-6563-6170-7368-637461726353"
version = "1.3.0"

[[deps.SentinelArrays]]
deps = ["Dates", "Random"]
git-tree-sha1 = "ebe7e59b37c400f694f52b58c93d26201387da70"
uuid = "91c51154-3ec4-41a3-a24f-3f23e20d615c"
version = "1.4.9"

[[deps.Serialization]]
uuid = "9e88b42a-f829-5b0c-bbe9-9e923198166b"

[[deps.Setfield]]
deps = ["ConstructionBase", "Future", "MacroTools", "StaticArraysCore"]
git-tree-sha1 = "c5391c6ace3bc430ca630251d02ea9687169ca68"
uuid = "efcf1570-3423-57d1-acb7-fd33fddbac46"
version = "1.1.2"

[[deps.Sockets]]
uuid = "6462fe0b-24de-5631-8697-dd941f90decc"

[[deps.SortingAlgorithms]]
deps = ["DataStructures"]
git-tree-sha1 = "64d974c2e6fdf07f8155b5b2ca2ffa9069b608d9"
uuid = "a2af1166-a08f-5f64-846c-94a0d3cef48c"
version = "1.2.2"

[[deps.SparseArrays]]
deps = ["Libdl", "LinearAlgebra", "Random", "Serialization", "SuiteSparse_jll"]
uuid = "2f01184e-e22b-5df5-ae63-d93ebab69eaf"
version = "1.10.0"

[[deps.SpecialFunctions]]
deps = ["IrrationalConstants", "LogExpFunctions", "OpenLibm_jll", "OpenSpecFun_jll"]
git-tree-sha1 = "5acc6a41b3082920f79ca3c759acbcecf18a8d78"
uuid = "276daf66-3868-5448-9aa4-cd146d93841b"
version = "2.7.1"

    [deps.SpecialFunctions.extensions]
    SpecialFunctionsChainRulesCoreExt = "ChainRulesCore"

    [deps.SpecialFunctions.weakdeps]
    ChainRulesCore = "d360d2e6-b24c-11e9-a2a3-2a2ae2dbcce4"

[[deps.StaticArraysCore]]
git-tree-sha1 = "6ab403037779dae8c514bad259f32a447262455a"
uuid = "1e83bf80-4336-4d27-bf5d-d5a4f845583c"
version = "1.4.4"

[[deps.Statistics]]
deps = ["LinearAlgebra", "SparseArrays"]
uuid = "10745b16-79ce-11e8-11f9-7d13ad32a3b2"
version = "1.10.0"

[[deps.StatsAPI]]
deps = ["LinearAlgebra"]
git-tree-sha1 = "178ed29fd5b2a2cfc3bd31c13375ae925623ff36"
uuid = "82ae8749-77ed-4fe6-ae5f-f523153014b0"
version = "1.8.0"

[[deps.StatsBase]]
deps = ["AliasTables", "DataAPI", "DataStructures", "IrrationalConstants", "LinearAlgebra", "LogExpFunctions", "Missings", "Printf", "Random", "SortingAlgorithms", "SparseArrays", "Statistics", "StatsAPI"]
git-tree-sha1 = "aceda6f4e598d331548e04cc6b2124a6148138e3"
uuid = "2913bbd2-ae8a-5f71-8c99-4fb6c76f3a91"
version = "0.34.10"

[[deps.StringManipulation]]
deps = ["PrecompileTools"]
git-tree-sha1 = "a3c1536470bf8c5e02096ad4853606d7c8f62721"
uuid = "892a3eda-7b42-436c-8928-eab12a02cf0e"
version = "0.4.2"

[[deps.StructUtils]]
deps = ["Dates", "UUIDs"]
git-tree-sha1 = "28145feabf717c5d65c1d5e09747ee7b1ff3ed13"
uuid = "ec057cc2-7a8d-4b58-b3b3-92acb9f63b42"
version = "2.6.3"

    [deps.StructUtils.extensions]
    StructUtilsMeasurementsExt = ["Measurements"]
    StructUtilsTablesExt = ["Tables"]

    [deps.StructUtils.weakdeps]
    Measurements = "eff96d63-e80a-5855-80a2-b1b0885c5ab7"
    Tables = "bd369af6-aec1-5ad0-b16a-f7cc5008161c"

[[deps.SuiteSparse_jll]]
deps = ["Artifacts", "Libdl", "libblastrampoline_jll"]
uuid = "bea87d4a-7f5b-5778-9afe-8cc45184846c"
version = "7.2.1+1"

[[deps.TOML]]
deps = ["Dates"]
uuid = "fa267f1f-6049-4f14-aa54-33bafae1ed76"
version = "1.0.3"

[[deps.TableTraits]]
deps = ["IteratorInterfaceExtensions"]
git-tree-sha1 = "c06b2f539df1c6efa794486abfb6ed2022561a39"
uuid = "3783bdb8-4a98-5b6b-af9a-565f29a5fe9c"
version = "1.0.1"

[[deps.Tables]]
deps = ["DataAPI", "DataValueInterfaces", "IteratorInterfaceExtensions", "OrderedCollections", "TableTraits"]
git-tree-sha1 = "f2c1efbc8f3a609aadf318094f8fc5204bdaf344"
uuid = "bd369af6-aec1-5ad0-b16a-f7cc5008161c"
version = "1.12.1"

[[deps.Tar]]
deps = ["ArgTools", "SHA"]
uuid = "a4e569a6-e804-4fa4-b0f3-eef7a1d5b13e"
version = "1.10.0"

[[deps.TensorCore]]
deps = ["LinearAlgebra"]
git-tree-sha1 = "1feb45f88d133a655e001435632f019a9a1bcdb6"
uuid = "62fd8b95-f654-4bbd-a8a5-9c27f68ccd50"
version = "0.1.1"

[[deps.Test]]
deps = ["InteractiveUtils", "Logging", "Random", "Serialization"]
uuid = "8dfed614-e22c-5e08-85e1-65c5234f0b40"

[[deps.TimeSeries]]
deps = ["Dates", "DelimitedFiles", "DocStringExtensions", "IteratorInterfaceExtensions", "PrettyTables", "RecipesBase", "Reexport", "Statistics", "TableTraits", "Tables"]
git-tree-sha1 = "e1235a1906d1e6d27c30c35daa91fcf6fc271f08"
uuid = "9e3dc215-6440-5c97-bce1-76c03772f85e"
version = "0.25.2"

[[deps.TranscodingStreams]]
git-tree-sha1 = "0c45878dcfdcfa8480052b6ab162cdd138781742"
uuid = "3bb67fe8-82b1-5028-8e26-92a6c54297fa"
version = "0.11.3"

[[deps.Tricks]]
git-tree-sha1 = "311349fd1c93a31f783f977a71e8b062a57d4101"
uuid = "410a4b4d-49e4-4fbc-ab6d-cb71b17b3775"
version = "0.1.13"

[[deps.URIs]]
git-tree-sha1 = "bef26fb046d031353ef97a82e3fdb6afe7f21b1a"
uuid = "5c2747f8-b7ea-4ff2-ba2e-563bfd36b1d4"
version = "1.6.1"

[[deps.UUIDs]]
deps = ["Random", "SHA"]
uuid = "cf7118a7-6976-5b1a-9a39-7adc72f591a4"

[[deps.UnPack]]
git-tree-sha1 = "387c1f73762231e86e0c9c5443ce3b4a0a9a0c2b"
uuid = "3a884ed6-31ef-47d7-9d2a-63182c4928ed"
version = "1.0.2"

[[deps.Unicode]]
uuid = "4ec0a83e-493e-50e2-b9ac-8f72acf5a8f5"

[[deps.WeakRefStrings]]
deps = ["DataAPI", "InlineStrings", "Parsers"]
git-tree-sha1 = "b1be2855ed9ed8eac54e5caff2afcdb442d52c23"
uuid = "ea10d353-3f73-51f8-a26c-33c1cb351aa5"
version = "1.4.2"

[[deps.WorkerUtilities]]
git-tree-sha1 = "cd1659ba0d57b71a464a29e64dbc67cfe83d54e7"
uuid = "76eceee3-57b5-4d4a-8e66-0e911cebbf60"
version = "1.6.1"

[[deps.Zlib_jll]]
deps = ["Libdl"]
uuid = "83775a58-1f1d-513f-b197-d71354ab007a"
version = "1.2.13+1"

[[deps.libblastrampoline_jll]]
deps = ["Artifacts", "Libdl"]
uuid = "8e850b90-86db-534c-a0d3-1478176c7d93"
version = "5.11.0+0"

[[deps.nghttp2_jll]]
deps = ["Artifacts", "Libdl"]
uuid = "8e850ede-7688-5339-a07c-302acd2aaf8d"
version = "1.52.0+1"

[[deps.p7zip_jll]]
deps = ["Artifacts", "Libdl"]
uuid = "3f19e933-33d8-53b3-aaab-bd5110c3b7a0"
version = "17.4.0+2"
"""

# ╔═╡ Cell order:
# ╟─576d5509-527d-4c48-9f2e-007fc977a0d8
# ╟─08782891-e0f5-4c25-9657-24212101724c
# ╟─67198764-1378-4286-8779-8a05011bf084
# ╟─9c11028c-8ccd-4c17-a964-86f09f785ba0
# ╟─e8f80d8c-5b6d-449a-8151-7c6f057492ae
# ╠═92c861a8-3027-4f39-a310-986aa47eea8f
# ╟─ad911687-8e0b-4340-81a6-452476fadf09
# ╟─775afb23-1849-444c-9c3f-abd5ef688d82
# ╟─523249f4-b761-48b8-9deb-dd72b26f6640
# ╟─25003d08-6b07-4c95-86dd-529ffbefe556
# ╟─f5c3e40f-38fe-45ec-b5e4-a6ea174939ee
# ╟─2663dfde-4e9a-4eb4-98c3-222c24b0f179
# ╟─3e834ad2-bc8e-46e4-9114-2c0836955994
# ╟─637413e8-c2d2-46d9-ac9c-9a57779bcc33
# ╟─28eff862-eb6a-4546-bfc7-f334aced0e5e
# ╟─734242df-1633-426a-acaa-977fe4706915
# ╠═11943aea-98c2-4dba-aa08-ab4241a06ffc
# ╠═fd2c78a8-0ee2-481d-9f60-289dd6a009af
# ╟─094df269-1140-4d12-bf0f-b22db7fbe677
# ╟─c502d1e7-65d9-410d-96fd-79999391e8fd
# ╟─51a0bef0-7b98-4513-9b83-63b1c944d9a8
# ╠═5e99d909-c2f3-4880-9855-edc3f791370e
# ╟─90a061e3-58f4-45f6-ba28-22d35b2aa9d5
# ╟─e522c316-57ca-4aa8-bd58-483f96e60e94
# ╟─93a211cf-5ad7-4fdc-a2fb-6981e61a0e3a
# ╟─964ccb85-d710-4de8-a647-5e60d21623de
# ╠═c7707bee-609b-4a8c-b154-747a279e8e21
# ╟─93580fd5-721b-4183-9e10-2d512eb9e772
# ╟─9e970d98-d334-4e4d-8ae8-fbd5587133e8
# ╟─d0208488-9671-4471-b90a-7eb3bb6a7623
# ╟─6f3bab31-f65f-4ca2-9e45-4cc3f630c608
# ╠═0f04c187-bc39-40dc-8832-d5d0f8b62120
# ╠═809d9a65-60f8-4281-9d3a-52ba40e913b8
# ╠═3b92d30c-4de2-4c50-9b36-5c9507bc1b71
# ╟─df63d7a9-57da-49b6-92e1-08e10c95a475
# ╟─61796e54-7c59-4f58-8c3b-d09d0652f074
# ╠═37b8be8f-9327-462a-974a-2c4bf5328880
# ╟─fdba7a56-ba94-4a1f-a235-b8ca465cf7a4
# ╠═9d9bee55-7cb2-41ce-81b8-6df112fdccec
# ╟─3da58f19-c84c-4e58-b6be-db439112e2ca
# ╟─72cd22a6-d335-454a-9791-f9847ef649c9
# ╟─6ee7f803-f923-4d6d-b381-8e2ea96d8772
# ╟─20f855aa-2199-42b5-bbb2-8c5cfcbf6b53
# ╟─a0a4cf82-0946-490a-9b89-ad931107d619
# ╟─b5c30c98-5757-4ad9-a4c6-129b0b69e88c
# ╟─d740937e-e748-4116-a351-633ae6630299
# ╟─d3296808-90cc-4c14-86ac-12718d1d9973
# ╟─793a93dc-c9d7-48ec-bc64-a4a67e710158
# ╠═d747423d-e45e-4b03-b604-f426b0ca64e4
# ╟─1092dad1-6c6a-4aef-a720-21826fa6c8f7
# ╟─c31fef9e-de1d-499f-a765-15f3db2a6dc0
# ╠═0f3fc6f9-f2d3-48a2-a0fb-4f8bcfddb84e
# ╟─9f3f14fc-aaed-4751-a054-382f978b3e6e
# ╟─9f4c0a5a-92dc-46bd-9f6a-d980d210cb3f
# ╟─476b34bc-da4d-4dbf-9143-fb1451ea7fd7
# ╟─55b995b4-84fc-445b-adcd-e3c65aea685d
# ╟─ae2ae896-87d0-43d6-af43-a5fada63c589
# ╟─667a85b6-7ceb-4d99-a9b9-5f66962479d1
# ╠═c05c3eb7-c0b6-4232-a2ca-21cf808a3234
# ╟─d349d5d5-4a68-4f67-b635-ee6ae39fc60a
# ╟─5d9920a1-ebc8-4d2c-974a-cc9106f286f2
# ╟─17242693-c3d4-4202-9e9e-332b5020638a
# ╠═3638dc40-f782-4804-8c62-d927e066e097
# ╟─241cb18f-21db-4f0f-aaf6-dc88a6a0d094
# ╟─8825b9f3-50d6-498f-b074-cd4496a9cd73
# ╠═81d80d65-4f62-4936-9b8f-bef79993bd8a
# ╠═f75f0124-171c-4519-abfe-e8361e95aac5
# ╟─0ba2e57a-76bd-4966-9260-8b4d1d9a81fb
# ╟─f37523c7-9171-46c8-9e81-f91343fa2a3a
# ╟─966c2fdc-0737-4fb7-9bf1-28d3df6ee7e4
# ╠═9332d9d9-9591-4587-8734-0a2cf1598286
# ╟─f97fa260-18d8-4f6a-94e1-46448b0d1842
# ╠═45b05a24-f860-434d-8047-5b6b86a3d2c6
# ╟─f0d2a99d-c287-4eb0-933d-4756a0e6a019
# ╟─94cfd9cf-586d-4a7a-b5ee-a0cb4c3ee288
# ╟─8c4238a8-2ca1-4a88-aee1-95c2afafb871
# ╟─88e1d017-2272-41a0-b97c-10ffc1c7f55b
# ╟─17e30d59-0856-42e1-bf47-20e6d0e37057
# ╟─9ad7d089-0885-4087-9ee0-4f9c35325441
# ╟─3311c9c9-3646-4bbb-adc5-1e09947808a1
# ╟─32ed885e-cf2f-4cee-b8df-cc81a463d429
# ╟─144c6ac4-4159-4520-bddd-3b75d624a426
# ╟─7bb3d405-e175-4865-a0a0-856a69aeb145
# ╟─92866e3b-5194-4ce7-81a3-e795b1b9a9a3
# ╟─23786f58-0f82-4a23-914c-0bb40c63f4e7
# ╟─5107f7cd-d9da-472a-bcba-9f9dd04107f7
# ╟─c1343361-86dd-447a-8ca1-9a2a506bf78d
# ╠═62c41a14-4cdd-488c-a653-0159f10fef1d
# ╟─4ae19cc8-4a3b-4118-af0a-d179f385916b
# ╟─7d2c6085-fd4b-42c7-a27c-b39f6ddf8290
# ╟─1e3710d8-8b8a-4e58-8d22-e6d5f2c5291e
# ╟─319d1622-6458-42a4-8e87-4aae991cab61
# ╠═9aa0ccd4-d986-4e6a-ba7b-5652e6edcc6f
# ╟─8eb4763a-6efb-49fd-a11d-59e724dd437c
# ╟─2fd32ff5-13f3-41cc-806e-d7e375a4c4ce
# ╟─243c8313-dfa2-4f96-ad95-2e9c815a15ea
# ╟─fd2c9069-9b3e-4b5f-8e1a-75728c43a387
# ╟─efeee7f2-b450-4db9-9a09-3c88b35252af
# ╟─ca074228-b0a1-40e6-8759-10791bd93ab0
# ╟─00000000-0000-0000-0000-000000000001
# ╟─00000000-0000-0000-0000-000000000002
